<?php

declare(strict_types=1);

namespace Akeneo\Tool\Bundle\MeasureBundle\Installer;

use Akeneo\Platform\Bundle\InstallerBundle\Event\InstallerEvents;
use Doctrine\DBAL\Driver\Connection;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * @author    Samir Boulil <samir.boulil@akeneo.com>
 * @copyright 2020 Akeneo SAS (http://www.akeneo.com)
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */
class MeasurementInstaller implements EventSubscriberInterface
{
    private Connection $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array
    {
        return [
            InstallerEvents::POST_DB_CREATE => ['createMeasurementTableAndStandardMeasurementFamilies'],
        ];
    }

    public function createMeasurementTableAndStandardMeasurementFamilies(): void
    {
        $this->createMeasurementTable();
        $this->loadStandardMeasurementFamilies();
    }

    private function loadStandardMeasurementFamilies(): void
    {
        $sql = <<<SQL
INSERT INTO `akeneo_measurement` (`code`, `labels`, `standard_unit`, `units`)
VALUES
    ('Area', '{"ca_ES":"Àrea","da_DK":"Areal","de_DE":"Fläche","en_GB":"Area","en_NZ":"Area","en_US":"Area","es_ES":"Superficie","fi_FI":"Alue","fr_FR":"Surface","it_IT":"Area","ja_JP":"エリア","pt_BR":"Área","ru_RU":"Площадь","sv_SE":"Område"}', 'SQUARE_METER', '[{"code":"SQUARE_MILLIMETER","labels":{"ca_ES":"Mil·límetre quadrat","da_DK":"Kvadrat millimeter","de_DE":"Quadratmillimeter","en_GB":"Square millimetre","en_NZ":"Square millimetre","en_US":"Square millimeter","es_ES":"Milímetro cuadrado","fi_FI":"Neliömillimetri","fr_FR":"Millimètre carré","it_IT":"Millimetro quadrato","ja_JP":"平方ミリメートル","pt_BR":"Milímetro quadrado","ru_RU":"Квадратный миллиметр","sv_SE":"Kvadratmillimeter"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"mm²"},{"code":"SQUARE_CENTIMETER","labels":{"ca_ES":"Centímetre quadrat","da_DK":"Kvadratcentimeter","de_DE":"Quadratzentimeter","en_GB":"Square centimetre","en_NZ":"Square centimetre","en_US":"Square centimeter","es_ES":"Centímetro cuadrado","fi_FI":"Neliösenttimetri","fr_FR":"Centimètre carré","it_IT":"Centimetro quadrato","ja_JP":"平方センチメートル","pt_BR":"Centímetro quadrado","ru_RU":"Квадратный сантиметр","sv_SE":"Kvadratcentimeter"},"convert_from_standard":[{"operator":"mul","value":"0.0001"}],"symbol":"cm²"},{"code":"SQUARE_DECIMETER","labels":{"ca_ES":"Decímetre quadrat","da_DK":"Kvadrat decimeter","de_DE":"Quadratdezimeter","en_GB":"Square decimetre","en_NZ":"Square decimetre","en_US":"Square decimeter","es_ES":"Decímetro cuadrado","fi_FI":"Neliödesimetri","fr_FR":"Décimètre carré","it_IT":"Decimetro quadrato","ja_JP":"平方デシメートル","pt_BR":"Decímetro quadrado","ru_RU":"Квадратный дециметр","sv_SE":"Kvadratdecimeter"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"dm²"},{"code":"SQUARE_METER","labels":{"ca_ES":"Metre quadrat","da_DK":"Kvadratmeter","de_DE":"Quadratmeter","en_GB":"Square metre","en_NZ":"Square metre","en_US":"Square meter","es_ES":"Metro cuadrado","fi_FI":"Neliömetri","fr_FR":"Mètre carré","it_IT":"Metro quadrato","ja_JP":"平方メートル","pt_BR":"Metro quadrado","ru_RU":"Квадратный метр","sv_SE":"Kvadratmeter"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"m²"},{"code":"CENTIARE","labels":{"ca_ES":"Centiàrees","da_DK":"Centiare","de_DE":"Quadratmeter","en_GB":"Centiare","en_NZ":"Centiare","en_US":"Centiare","es_ES":"Centiáreas","fi_FI":"Senttiaari","fr_FR":"Centiare","it_IT":"Centiara","ja_JP":"センチアール","pt_BR":"Centiare","ru_RU":"Центнер","sv_SE":"Kvadratmeter"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"ca"},{"code":"SQUARE_DEKAMETER","labels":{"ca_ES":"Decàmetre quadrat","da_DK":"Kvadrat dekameter","de_DE":"Quadratdekameter","en_GB":"Square decametre","en_NZ":"Square dekametre","en_US":"Square dekameter","es_ES":"Dekametro cuadrado","fi_FI":"Neliödekametri","fr_FR":"Décamètre carré","it_IT":"Decametro quadrato","ja_JP":"平方デカメートル","pt_BR":"Decametro quadrado","ru_RU":"Квадратный декаметр","sv_SE":"Kvadratdekameter"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"dam²"},{"code":"ARE","labels":{"ca_ES":"Àrea","da_DK":"Are","de_DE":"Ar","en_GB":"Sú","en_NZ":"Are","en_US":"Are","es_ES":"Área","fi_FI":"Aari","fr_FR":"Are","it_IT":"Ara","ja_JP":"アール","pt_BR":"Area","ru_RU":"Ар","sv_SE":"Hektar"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"a"},{"code":"SQUARE_HECTOMETER","labels":{"ca_ES":"Hectòmetre quadrat","da_DK":"Kvadrat hectometer","de_DE":"Quadrathektometer","en_GB":"Square hectometre","en_NZ":"Square hectometre","en_US":"Square hectometer","es_ES":"Hectómetro cuadrado","fi_FI":"Neliöhehtometri","fr_FR":"Hectomètre carré","it_IT":"Ettometro quadrato","ja_JP":"平方ヘクトメートル","pt_BR":"Hectómetro quadrado","ru_RU":"Квадратный гектометр","sv_SE":"Kvadrathektameter"},"convert_from_standard":[{"operator":"mul","value":"10000"}],"symbol":"hm²"},{"code":"HECTARE","labels":{"ca_ES":"Hectàrees","da_DK":"Hektar","de_DE":"Hektar","en_GB":"Hectare","en_NZ":"Hectare","en_US":"Hectare","es_ES":"Hectárea","fi_FI":"Hehtaari","fr_FR":"Hectare","it_IT":"Ettaro","ja_JP":"ヘクタール","pt_BR":"Hectare","ru_RU":"Гектар","sv_SE":"Hektar"},"convert_from_standard":[{"operator":"mul","value":"10000"}],"symbol":"ha"},{"code":"SQUARE_KILOMETER","labels":{"ca_ES":"Quilòmetre quadrat","da_DK":"Kvadrat kilometer","de_DE":"Quadratkilometer","en_GB":"Square kilometre","en_NZ":"Square kilometre","en_US":"Square kilometer","es_ES":"Kilómetro cuadrado","fi_FI":"Neliökilometri","fr_FR":"Kilomètre carré","it_IT":"Chilometro quadrato","ja_JP":"平方キロメートル","pt_BR":"Quilômetro quadrado","ru_RU":"Квадратный километр","sv_SE":"Kvadratkilometer"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"km²"},{"code":"SQUARE_MIL","labels":{"ca_ES":"Mil quadrat","da_DK":"Kvadrat mil","de_DE":"Quadratmil","en_GB":"Square mil","en_NZ":"Square mil","en_US":"Square mil","es_ES":"Mil cuadrado","fi_FI":"Neliötuhannesosatuuma","fr_FR":"Mil carré","it_IT":"Mil quadrati","ja_JP":"平方ミル","pt_BR":"Mil quadrada","ru_RU":"Квадратная миля","sv_SE":"Kvadratmil"},"convert_from_standard":[{"operator":"mul","value":"0.00000000064516"}],"symbol":"sq mil"},{"code":"SQUARE_INCH","labels":{"ca_ES":"Polzada quadrada","da_DK":"Kvadrattomme","de_DE":"Quadratzoll","en_GB":"Square inch","en_NZ":"Square inch","en_US":"Square inch","es_ES":"Pulgada cuadrada","fi_FI":"Neliötuuma","fr_FR":"Pouce carré","it_IT":"Pollice quadrato","ja_JP":"平方インチ","pt_BR":"Polegada quadrada","ru_RU":"Квадратный дюйм","sv_SE":"Kvadrattum"},"convert_from_standard":[{"operator":"mul","value":"0.00064516"}],"symbol":"in²"},{"code":"SQUARE_FOOT","labels":{"ca_ES":"Peu quadrat","da_DK":"Kvadratfod","de_DE":"Quadratfuß","en_GB":"Square foot","en_NZ":"Square foot","en_US":"Square foot","es_ES":"Pies cuadrados","fi_FI":"Neliöjalka","fr_FR":"Pied carré","it_IT":"Piede quadrato","ja_JP":"平方フィート","pt_BR":"Pé quadrado","ru_RU":"Квадратный фут","sv_SE":"Kvadratfot"},"convert_from_standard":[{"operator":"mul","value":"0.09290304"}],"symbol":"ft²"},{"code":"SQUARE_YARD","labels":{"ca_ES":"Iarda quadrada","da_DK":"Kvadrat yard","de_DE":"Quadratyard","en_GB":"Square yard","en_NZ":"Square yard","en_US":"Square yard","es_ES":"Yarda cuadrada","fi_FI":"Neliöjaardi","fr_FR":"Yard carré","it_IT":"Yard quadrata","ja_JP":"平方ヤード","pt_BR":"Jarda quadrada","ru_RU":"Квадратный ярд","sv_SE":"Kvadratyard"},"convert_from_standard":[{"operator":"mul","value":"0.83612736"}],"symbol":"yd²"},{"code":"ARPENT","labels":{"ca_ES":"Arpent","da_DK":"Arpent","de_DE":"Arpent","en_GB":"Arpent","en_NZ":"Arpent","en_US":"Arpent","es_ES":"Arpende","fi_FI":"Eekkeri","fr_FR":"Arpent","it_IT":"Arpenti","ja_JP":"アルパン","pt_BR":"Arpent","ru_RU":"Арпан","sv_SE":"Arpent"},"convert_from_standard":[{"operator":"mul","value":"3418.89"}],"symbol":"arpent"},{"code":"ACRE","labels":{"ca_ES":"Acre","da_DK":"Tønder","de_DE":"Morgen","en_GB":"Acre","en_NZ":"Acre","en_US":"Acre","es_ES":"Acre","fi_FI":"Eekkeri","fr_FR":"Acre","it_IT":"Acri","ja_JP":"エーカー","pt_BR":"Acre","ru_RU":"Акр","sv_SE":"Tunnland"},"convert_from_standard":[{"operator":"mul","value":"4046.856422"}],"symbol":"A"},{"code":"SQUARE_FURLONG","labels":{"ca_ES":"Furlong quadrat","da_DK":"Kvadratisk furlong","de_DE":"Quadrat-Achtelmeile","en_GB":"Square furlong","en_NZ":"Square furlong","en_US":"Square furlong","es_ES":"Estadio cuadrado","fi_FI":"Vakomitta","fr_FR":"Furlong carré","it_IT":"Furlong quadrato","ja_JP":"平方ハロン","pt_BR":"Furlong quadrado","ru_RU":"Квадратный фурлонг","sv_SE":"Kvadratfurlong"},"convert_from_standard":[{"operator":"mul","value":"40468.726"}],"symbol":"fur²"},{"code":"SQUARE_MILE","labels":{"ca_ES":"Milla quadrada","da_DK":"Kvadrat mil","de_DE":"Quadratmeile","en_GB":"Square mile","en_NZ":"Square mile","en_US":"Square mile","es_ES":"Milla cuadrada","fi_FI":"Neliömaili","fr_FR":"Mile carré","it_IT":"Miglio quadrato","ja_JP":"平方マイル","pt_BR":"Milha quadrada","ru_RU":"Квадратная миля","sv_SE":"Kvadratmile"},"convert_from_standard":[{"operator":"mul","value":"2589988.110336"}],"symbol":"mi²"}]'),
    ('Binary', '{"ca_ES":"Binari","da_DK":"Binær","de_DE":"Binär","en_GB":"Binary","en_NZ":"Binary","en_US":"Binary","es_ES":"Binario","fi_FI":"Binääri","fr_FR":"Binaire","it_IT":"Binario","ja_JP":"バイナリ","pt_BR":"Binário","ru_RU":"Двоичный","sv_SE":"Binär"}', 'BYTE', '[{"code":"CHAR","labels":{"fr_FR":"Char","en_US":"Char"},"convert_from_standard":[{"operator":"mul","value":"8"}],"symbol":"char"},{"code":"KILOBIT","labels":{"fr_FR":"Kilobit","en_US":"Kilobit"},"convert_from_standard":[{"operator":"mul","value":"125"}],"symbol":"kbit"},{"code":"MEGABIT","labels":{"fr_FR":"Mégabit","en_US":"Megabit"},"convert_from_standard":[{"operator":"mul","value":"125000"}],"symbol":"Mbit"},{"code":"GIGABIT","labels":{"fr_FR":"Gigabit","en_US":"Gigabit"},"convert_from_standard":[{"operator":"mul","value":"125000000"}],"symbol":"Gbit"},{"code":"BIT","labels":{"ca_ES":"Bit","da_DK":"Bit","de_DE":"Bit","en_GB":"Bit","en_NZ":"Bit","en_US":"Bit","es_ES":"Bit","fi_FI":"Bitti","fr_FR":"Bit","it_IT":"Bit","ja_JP":"ビット","pt_BR":"Bit","ru_RU":"Бит","sv_SE":"Bit"},"convert_from_standard":[{"operator":"mul","value":"0.125"}],"symbol":"b"},{"code":"BYTE","labels":{"ca_ES":"Byte","da_DK":"Byte","de_DE":"Byte","en_GB":"Byte","en_NZ":"Byte","en_US":"Byte","es_ES":"Byte","fi_FI":"Tavu","fr_FR":"Octet","it_IT":"Byte","ja_JP":"バイト","pt_BR":"Byte","ru_RU":"Байт","sv_SE":"Byte"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"B"},{"code":"KILOBYTE","labels":{"ca_ES":"Kilobyte","da_DK":"Kilobyte","de_DE":"Kilobyte","en_GB":"Kilobyte","en_NZ":"Kilobyte","en_US":"Kilobyte","es_ES":"Kilobyte","fi_FI":"Kilotavu","fr_FR":"Kilo-octet","it_IT":"Kilobyte","ja_JP":"キロバイト","pt_BR":"Kilobyte (KB)","ru_RU":"Килобайт","sv_SE":"Kilobyte"},"convert_from_standard":[{"operator":"mul","value":"1024"}],"symbol":"kB"},{"code":"MEGABYTE","labels":{"ca_ES":"Megabyte","da_DK":"Megabyte","de_DE":"Megabyte","en_GB":"Megabyte","en_NZ":"Megabyte","en_US":"Megabyte","es_ES":"Megabyte","fi_FI":"Megatavu","fr_FR":"Mégaoctet","it_IT":"Megabyte","ja_JP":"メガバイト","pt_BR":"Megabyte (MB)","ru_RU":"Мегабайт","sv_SE":"Megabyte"},"convert_from_standard":[{"operator":"mul","value":"1048576"}],"symbol":"MB"},{"code":"GIGABYTE","labels":{"ca_ES":"Gigabyte","da_DK":"Gigabyte","de_DE":"Gigabyte","en_GB":"Gigabyte","en_NZ":"Gigabyte","en_US":"Gigabyte","es_ES":"Gigabyte","fi_FI":"Gigatavu","fr_FR":"Gigaoctet","it_IT":"Gigabyte","ja_JP":"ギガバイト","pt_BR":"Gigabyte (GB)","ru_RU":"Гигабайт","sv_SE":"Gigabyte"},"convert_from_standard":[{"operator":"mul","value":"1073741824"}],"symbol":"GB"},{"code":"TERABYTE","labels":{"ca_ES":"Terabyte","da_DK":"Terabyte","de_DE":"Terabyte","en_GB":"Terabyte","en_NZ":"Terabyte","en_US":"Terabyte","es_ES":"Terabyte","fi_FI":"Teratavu","fr_FR":"Téraoctet","it_IT":"Terabyte","ja_JP":"テラバイト","pt_BR":"Terabyte (TB)","ru_RU":"Терабайт","sv_SE":"Terabyte"},"convert_from_standard":[{"operator":"mul","value":"1099511627776"}],"symbol":"TB"}]'),
    ('Decibel', '{"ca_ES":"Decibel","da_DK":"Decibel","de_DE":"Dezibel","en_GB":"Decibel","en_NZ":"Decibel","en_US":"Decibel","es_ES":"Decibelio","fi_FI":"Desibeli","fr_FR":"Décibel","it_IT":"Decibel","ja_JP":"デシベル","pt_BR":"Decibel","ru_RU":"Децибел","sv_SE":"Decibel"}', 'DECIBEL', '[{"code":"DECIBEL","labels":{"ca_ES":"Decibel","da_DK":"Decibel","de_DE":"Dezibel","en_GB":"Decibel","en_NZ":"Decibel","en_US":"Decibel","es_ES":"Decibelio","fi_FI":"Desibeli","fr_FR":"Décibel","it_IT":"Decibel","ja_JP":"デシベル","pt_BR":"Decibel","ru_RU":"Децибел","sv_SE":"Decibel"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"dB"}]'),
    ('Frequency', '{"ca_ES":"Freqüència","da_DK":"Frekvens","de_DE":"Frequenz","en_GB":"Frequency","en_NZ":"Frequency","en_US":"Frequency","es_ES":"Frecuencia","fi_FI":"Frekvenssi","fr_FR":"Fréquence","it_IT":"Frequenza","ja_JP":"頻度","pt_BR":"Frequência","ru_RU":"Частота","sv_SE":"Frekvens"}', 'HERTZ', '[{"code":"HERTZ","labels":{"ca_ES":"Hertz","da_DK":"Hertz","de_DE":"Hertz","en_GB":"Hertz","en_NZ":"Hertz","en_US":"Hertz","es_ES":"Hercios","fi_FI":"Hertsi","fr_FR":"Hertz","it_IT":"Hertz","ja_JP":"ヘルツ","pt_BR":"Hertz (Hz)","ru_RU":"Герц","sv_SE":"Hertz"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"Hz"},{"code":"KILOHERTZ","labels":{"ca_ES":"Kilohertz","da_DK":"Kilohertz","de_DE":"Kilohertz","en_GB":"Kilohertz","en_NZ":"Kilohertz","en_US":"Kilohertz","es_ES":"Kilohercios","fi_FI":"Kilohertsi","fr_FR":"Kilohertz","it_IT":"Kilohertz","ja_JP":"キロヘルツ","pt_BR":"Kilohertz (kHz)","ru_RU":"Килогерц","sv_SE":"Kilohertz"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kHz"},{"code":"MEGAHERTZ","labels":{"ca_ES":"Megahertz","da_DK":"Megahertz","de_DE":"Megahertz","en_GB":"Megahertz","en_NZ":"Megahertz","en_US":"Megahertz","es_ES":"Megahercios","fi_FI":"Megahertsi","fr_FR":"Mégahertz","it_IT":"Megahertz","ja_JP":"メガヘルツ","pt_BR":"Megahertz (MHz)","ru_RU":"Мегагерц","sv_SE":"Megahertz"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"MHz"},{"code":"GIGAHERTZ","labels":{"ca_ES":"Gigahertz","da_DK":"Gigahertz","de_DE":"Gigahertz","en_GB":"Gigahertz","en_NZ":"Gigahertz","en_US":"Gigahertz","es_ES":"Gigahercios","fi_FI":"Gigahertsi","fr_FR":"Gigahertz","it_IT":"Gigahertz","ja_JP":"ギガヘルツ","pt_BR":"Gigahertz (GHz)","ru_RU":"Гигагерц","sv_SE":"Gigahertz"},"convert_from_standard":[{"operator":"mul","value":"1000000000"}],"symbol":"GHz"},{"code":"TERAHERTZ","labels":{"ca_ES":"Terahertz","da_DK":"Terahertz","de_DE":"Terahertz","en_GB":"Terahertz","en_NZ":"Terahertz","en_US":"Terahertz","es_ES":"Terahercios","fi_FI":"Terahertsi","fr_FR":"Térahertz","it_IT":"Terahertz","ja_JP":"テラヘルツ","pt_BR":"Terahertz (THz)","ru_RU":"Терагерц","sv_SE":"Terahertz"},"convert_from_standard":[{"operator":"mul","value":"1000000000000"}],"symbol":"THz"}]'),
    ('Length', '{"ca_ES":"Longitud","da_DK":"Længde","de_DE":"Länge","en_GB":"Length","en_NZ":"Length","en_US":"Length","es_ES":"Longitud","fi_FI":"Pituus","fr_FR":"Longueur","it_IT":"Lunghezza","ja_JP":"長さ","pt_BR":"Comprimento","ru_RU":"Длина","sv_SE":"Längd"}', 'METER', '[{"code":"MICROMETER","labels":{"fr_FR":"Micromètre","en_US":"Micrometer"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"μm"},{"code":"NAUTICAL_MILE","labels":{"fr_FR":"Mile nautique","en_US":"Nautical mile"},"convert_from_standard":[{"operator":"mul","value":"1852"}],"symbol":"nm"},{"code":"MILLIMETER","labels":{"ca_ES":"Mil·límetre","da_DK":"Millimeter","de_DE":"Millimeter","en_GB":"Millimetre","en_NZ":"Millimetre","en_US":"Millimeter","es_ES":"Milímetro","fi_FI":"Millimetri","fr_FR":"Millimètre","it_IT":"Millimetro","ja_JP":"ミリメートル","pt_BR":"Milímetro","ru_RU":"Миллиметр","sv_SE":"Millimeter"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mm"},{"code":"CENTIMETER","labels":{"ca_ES":"Centímetre","da_DK":"Centimeter","de_DE":"Zentimeter","en_GB":"Centimetre","en_NZ":"Centimetre","en_US":"Centimeter","es_ES":"Centímetro","fi_FI":"Senttimetri","fr_FR":"Centimètre","it_IT":"Centimetro","ja_JP":"センチメートル","pt_BR":"Centímetro","ru_RU":"Сантиметр","sv_SE":"Centimeter"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"cm"},{"code":"DECIMETER","labels":{"ca_ES":"Decímetre","da_DK":"Decimeter","de_DE":"Dezimeter","en_GB":"Decimetre","en_NZ":"Decimetre","en_US":"Decimeter","es_ES":"Decímetro","fi_FI":"Desimetri","fr_FR":"Décimètre","it_IT":"Decimetro","ja_JP":"デシメートル","pt_BR":"Decímetro","ru_RU":"Дециметр","sv_SE":"Decimeter"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"dm"},{"code":"METER","labels":{"ca_ES":"Metre","da_DK":"Meter","de_DE":"Meter","en_GB":"Metre","en_NZ":"Metre","en_US":"Meter","es_ES":"Medidor","fi_FI":"Metri","fr_FR":"Mètre","it_IT":"Metro","ja_JP":"メートル","pt_BR":"Metro","ru_RU":"Метр","sv_SE":"Meter"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"m"},{"code":"DEKAMETER","labels":{"ca_ES":"Decàmetre","da_DK":"Dekameter","de_DE":"Dekameter","en_GB":"Decametre","en_NZ":"Dekametre","en_US":"Dekameter","es_ES":"Decámetro","fi_FI":"Dekametri","fr_FR":"Décamètre","it_IT":"Decametro","ja_JP":"デカメートル","pt_BR":"Decâmetro","ru_RU":"Декаметр","sv_SE":"Dekameter"},"convert_from_standard":[{"operator":"mul","value":"10"}],"symbol":"dam"},{"code":"HECTOMETER","labels":{"ca_ES":"Hectòmetre","da_DK":"Hectometer","de_DE":"Hektometer","en_GB":"Hectometre","en_NZ":"Hectometre","en_US":"Hectometer","es_ES":"Hectómetro","fi_FI":"Hehtometri","fr_FR":"Hectomètre","it_IT":"Ettometro","ja_JP":"ヘクトメートル","pt_BR":"Hectómetro","ru_RU":"Гектометр","sv_SE":"Hektometer"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"hm"},{"code":"KILOMETER","labels":{"ca_ES":"Quilòmetre","da_DK":"Kilometer","de_DE":"Kilometer","en_GB":"Kilometre","en_NZ":"Kilometre","en_US":"Kilometer","es_ES":"Kilómetro","fi_FI":"Kilometri","fr_FR":"Kilomètre","it_IT":"Chilometro","ja_JP":"キロメートル","pt_BR":"Quilômetro","ru_RU":"Километр","sv_SE":"Kilometer"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"km"},{"code":"MIL","labels":{"ca_ES":"Mil","da_DK":"Mil","de_DE":"Mil","en_GB":"Mil","en_NZ":"Mil","en_US":"Mil","es_ES":"Mil","fi_FI":"Peninkulma","fr_FR":"Mil","it_IT":"Mil","ja_JP":"ミル","pt_BR":"Mil","ru_RU":"Миля","sv_SE":"Mil"},"convert_from_standard":[{"operator":"mul","value":"0.0000254"}],"symbol":"mil"},{"code":"INCH","labels":{"ca_ES":"Polzada","da_DK":"Tommer","de_DE":"Zoll","en_GB":"Inch","en_NZ":"Inch","en_US":"Inch","es_ES":"Pulgadas","fi_FI":"Tuuma","fr_FR":"Pouce","it_IT":"Pollici","ja_JP":"インチ","pt_BR":"Polegada","ru_RU":"Дюйм","sv_SE":"Tum"},"convert_from_standard":[{"operator":"mul","value":"0.0254"}],"symbol":"in"},{"code":"FEET","labels":{"ca_ES":"Peus","da_DK":"Fod","de_DE":"Fuß","en_GB":"Feet","en_NZ":"Feet","en_US":"Feet","es_ES":"Pies","fi_FI":"Jalka","fr_FR":"Pieds","it_IT":"Piedi","ja_JP":"フィート","pt_BR":"Pé","ru_RU":"Фут","sv_SE":"Fot"},"convert_from_standard":[{"operator":"mul","value":"0.3048"}],"symbol":"ft"},{"code":"YARD","labels":{"ca_ES":"Iarda","da_DK":"Yard","de_DE":"Yard","en_GB":"Yard","en_NZ":"Yard","en_US":"Yard","es_ES":"Yarda","fi_FI":"Jaardi","fr_FR":"Yard","it_IT":"Yard","ja_JP":"ヤード","pt_BR":"Jarda","ru_RU":"Ярд","sv_SE":"Yard"},"convert_from_standard":[{"operator":"mul","value":"0.9144"}],"symbol":"yd"},{"code":"CHAIN","labels":{"ca_ES":"Cadena","da_DK":"Kæde","de_DE":"Kette","en_GB":"Chain","en_NZ":"Chain","en_US":"Chain","es_ES":"Cadena","fi_FI":"Mittaketju","fr_FR":"Chaîne","it_IT":"Chain","ja_JP":"チェーン","pt_BR":"Cadeia","ru_RU":"Чейн","sv_SE":"Chain"},"convert_from_standard":[{"operator":"mul","value":"20.1168"}],"symbol":"ch"},{"code":"FURLONG","labels":{"ca_ES":"Furlong","da_DK":"Furlong","de_DE":"Achtelmeile","en_GB":"Furlong","en_NZ":"Furlong","en_US":"Furlong","es_ES":"Estadio","fi_FI":"Vakomitta","fr_FR":"Furlong","it_IT":"Furlong","ja_JP":"ハロン","pt_BR":"Furlong","ru_RU":"Фурлонг","sv_SE":"Furlong"},"convert_from_standard":[{"operator":"mul","value":"201.168"}],"symbol":"fur"},{"code":"MILE","labels":{"ca_ES":"Milla","da_DK":"Mil","de_DE":"Meile","en_GB":"Mile","en_NZ":"Mile","en_US":"Mile","es_ES":"Milla","fi_FI":"Maili","fr_FR":"Mile","it_IT":"Miglio","ja_JP":"マイル","pt_BR":"Milha","ru_RU":"Миля","sv_SE":"Engelsk mil"},"convert_from_standard":[{"operator":"mul","value":"1609.344"}],"symbol":"mi"}]'),
    ('Power', '{"ca_ES":"Potència","da_DK":"Effekt","de_DE":"Leistung","en_GB":"Power","en_NZ":"Power","en_US":"Power","es_ES":"Potencia","fi_FI":"Voima","fr_FR":"Puissance","it_IT":"Potenza","ja_JP":"電源","pt_BR":"Potência","ru_RU":"Мощность","sv_SE":"Ström"}', 'WATT', '[{"code":"WATT","labels":{"ca_ES":"Watt","da_DK":"Watt","de_DE":"Watt","en_GB":"Watt","en_NZ":"Watt","en_US":"Watt","es_ES":"Vatio","fi_FI":"Watti","fr_FR":"Watt","it_IT":"Watt","ja_JP":"ワット","pt_BR":"Watt","ru_RU":"Ватт","sv_SE":"Watt"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"W"},{"code":"KILOWATT","labels":{"ca_ES":"Quilowatt","da_DK":"Kilowatt","de_DE":"Kilowatt","en_GB":"Kilowatt","en_NZ":"Kilowatt","en_US":"Kilowatt","es_ES":"Kilovatio","fi_FI":"Kilowatti","fr_FR":"Kilowatt","it_IT":"Chilowatt","ja_JP":"キロワット","pt_BR":"Kilowatt","ru_RU":"Киловатт","sv_SE":"Kilowatt"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kW"},{"code":"MEGAWATT","labels":{"ca_ES":"Megawatt","da_DK":"Megawatt","de_DE":"Megawatt","en_GB":"Megawatt","en_NZ":"Megawatt","en_US":"Megawatt","es_ES":"Megavatio","fi_FI":"Megawatti","fr_FR":"Mégawatt","it_IT":"Megawatt","ja_JP":"メガワット","pt_BR":"Megawatt","ru_RU":"Мегаватт","sv_SE":"Megawatt"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"MW"},{"code":"GIGAWATT","labels":{"ca_ES":"Gigawatt","da_DK":"Gigawatt","de_DE":"Gigawatt","en_GB":"Gigawatt","en_NZ":"Gigawatt","en_US":"Gigawatt","es_ES":"Gigavatios","fi_FI":"Gigawatti","fr_FR":"Gigawatt","it_IT":"Gigawatt","ja_JP":"ギガワット","pt_BR":"Gigawatt","ru_RU":"Гигаватт","sv_SE":"Gigawatt"},"convert_from_standard":[{"operator":"mul","value":"1000000000"}],"symbol":"GW"},{"code":"TERAWATT","labels":{"ca_ES":"Terawatt","da_DK":"Terawatt","de_DE":"Terawatt","en_GB":"Terawatt","en_NZ":"Terawatt","en_US":"Terawatt","es_ES":"Teravatios","fi_FI":"Terawatti","fr_FR":"Térawatt","it_IT":"Terawatt","ja_JP":"テラワット","pt_BR":"Terawatt","ru_RU":"Тераватт","sv_SE":"Terawatt"},"convert_from_standard":[{"operator":"mul","value":"1000000000000"}],"symbol":"TW"}]'),
    ('Voltage', '{"ca_ES":"Voltatge","da_DK":"Spænding","de_DE":"Spannung","en_GB":"Voltage","en_NZ":"Voltage","en_US":"Voltage","es_ES":"Voltaje","fi_FI":"Jännite","fr_FR":"Tension","it_IT":"Tensione elettrica","ja_JP":"電圧","pt_BR":"Voltagem","ru_RU":"Напряжение","sv_SE":"Spänning"}', 'VOLT', '[{"code":"MILLIVOLT","labels":{"ca_ES":"Mil·livolt","da_DK":"Millivolt","de_DE":"Millivolt","en_GB":"Millivolt","en_NZ":"Millivolt","en_US":"Millivolt","es_ES":"Milivoltio","fi_FI":"Millivoltti","fr_FR":"Millivolt","it_IT":"Millivolt","ja_JP":"ミリボルト","pt_BR":"Millivolt","ru_RU":"Милливольт","sv_SE":"Millivolt"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mV"},{"code":"CENTIVOLT","labels":{"ca_ES":"Centivolt","da_DK":"Centivolt","de_DE":"Centivolt","en_GB":"Centivolt","en_NZ":"Centivolt","en_US":"Centivolt","es_ES":"Centivoltio","fi_FI":"Senttivoltti","fr_FR":"Centivolt","it_IT":"Centivolt","ja_JP":"センチボルト","pt_BR":"Centivolt","ru_RU":"Сантивольт","sv_SE":"Centivolt"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"cV"},{"code":"DECIVOLT","labels":{"ca_ES":"Decivolt","da_DK":"Decivolt","de_DE":"Decivolt","en_GB":"Decivolt","en_NZ":"Decivolt","en_US":"Decivolt","es_ES":"Decivoltio","fi_FI":"Desivoltti","fr_FR":"Décivolt","it_IT":"Decivolt","ja_JP":"デシボルト","pt_BR":"Decivolt","ru_RU":"Децивольт","sv_SE":"Decivolt"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"dV"},{"code":"VOLT","labels":{"ca_ES":"Volt","da_DK":"Volt","de_DE":"Volt","en_GB":"Volt","en_NZ":"Volt","en_US":"Volt","es_ES":"Voltio","fi_FI":"Voltti","fr_FR":"Volt","it_IT":"Volt","ja_JP":"ボルト","pt_BR":"Volt","ru_RU":"Вольт","sv_SE":"Volt"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"V"},{"code":"DEKAVOLT","labels":{"ca_ES":"Decavolt","da_DK":"Dekavolt","de_DE":"Dekavolt","en_GB":"Dekavolt","en_NZ":"Dekavolt","en_US":"Dekavolt","es_ES":"Decavoltio","fi_FI":"Dekavoltti","fr_FR":"Décavolt","it_IT":"Decavolt","ja_JP":"デカボルト","pt_BR":"Dekavolt","ru_RU":"Декавольт","sv_SE":"Dekavolt"},"convert_from_standard":[{"operator":"mul","value":"10"}],"symbol":"daV"},{"code":"HECTOVOLT","labels":{"ca_ES":"Hectovolt","da_DK":"Hectovolt","de_DE":"Hectovolt","en_GB":"Hectovolt","en_NZ":"Hectovolt","en_US":"Hectovolt","es_ES":"Hectovoltio","fi_FI":"Hehtovoltti","fr_FR":"Hectovolt","it_IT":"Hectovolt","ja_JP":"ヘクトボルト","pt_BR":"Hectovolt","ru_RU":"Гектовольт","sv_SE":"Hectovolt"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"hV"},{"code":"KILOVOLT","labels":{"ca_ES":"Quilovolt","da_DK":"Kilovolt","de_DE":"Kilovolt","en_GB":"Kilovolt","en_NZ":"Kilovolt","en_US":"Kilovolt","es_ES":"Kilovoltio","fi_FI":"Kilovoltti","fr_FR":"Kilovolt","it_IT":"Kilovolt","ja_JP":"キロボルト","pt_BR":"Kilovolt","ru_RU":"Киловольт","sv_SE":"Kilovolt"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kV"}]'),
    ('Intensity', '{"ca_ES":"Intensitat","da_DK":"Intensitet","de_DE":"Intensität","en_GB":"Intensity","en_NZ":"Intensity","en_US":"Intensity","es_ES":"Intensidad","fi_FI":"Intensiteetti","fr_FR":"Intensité","it_IT":"Intensità","ja_JP":"強度","pt_BR":"Intensidade","ru_RU":"Интенсивность","sv_SE":"Intensitet"}', 'AMPERE', '[{"code":"MILLIAMPERE","labels":{"ca_ES":"Mil·liamper","da_DK":"Milliampere","de_DE":"Milliampere","en_GB":"Milliampere","en_NZ":"Milliampere","en_US":"Milliampere","es_ES":"Miliamperio","fi_FI":"Milliampeeri","fr_FR":"Milliampères","it_IT":"Milliampere","ja_JP":"ミリ アンペア","pt_BR":"Miliampère","ru_RU":"Миллиампер","sv_SE":"Milliampere"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mA"},{"code":"CENTIAMPERE","labels":{"ca_ES":"Centiamper","da_DK":"Centiampere","de_DE":"Centiampere","en_GB":"Centiampere","en_NZ":"Centiampere","en_US":"Centiampere","es_ES":"Centiamperio","fi_FI":"Senttiampeeri","fr_FR":"Centiampere","it_IT":"Centiampere","ja_JP":"センチアンペア","pt_BR":"Centiampere","ru_RU":"Сантиампер","sv_SE":"Centiampere"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"cA"},{"code":"DECIAMPERE","labels":{"ca_ES":"Deciamper","da_DK":"Deciampere","de_DE":"Deciampere","en_GB":"Deciampere","en_NZ":"Deciampere","en_US":"Deciampere","es_ES":"Deciamperio","fi_FI":"Desiampeeri","fr_FR":"Deciampere","it_IT":"Deciampere","ja_JP":"デシアンペア","pt_BR":"Deciampere","ru_RU":"Дециампер","sv_SE":"Deciampere"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"dA"},{"code":"AMPERE","labels":{"ca_ES":"Amper","da_DK":"Ampere","de_DE":"Ampere","en_GB":"Ampere","en_NZ":"Ampere","en_US":"Ampere","es_ES":"Amperio","fi_FI":"Ampeeri","fr_FR":"Ampère","it_IT":"Ampere","ja_JP":"アンペア","pt_BR":"Ampère","ru_RU":"Ампер","sv_SE":"Ampere"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"A"},{"code":"DEKAMPERE","labels":{"ca_ES":"Decaamper","da_DK":"Dekaampere","de_DE":"Dekaampere","en_GB":"Dekaampere","en_NZ":"Dekaampere","en_US":"Dekaampere","es_ES":"Decaamperio","fi_FI":"Deka-ampeeri","fr_FR":"Décaampere","it_IT":"Decaampere","ja_JP":"デカアンペア","pt_BR":"Dekaampere","ru_RU":"Декаампер","sv_SE":"Dekaampere"},"convert_from_standard":[{"operator":"mul","value":"10"}],"symbol":"daA"},{"code":"HECTOAMPERE","labels":{"ca_ES":"Hectoamper","da_DK":"Hectoampere","de_DE":"Hectoampere","en_GB":"Hectoampere","en_NZ":"Hectoampere","en_US":"Hectoampere","es_ES":"Hectoamperio","fi_FI":"Hehtoampeeri","fr_FR":"Hectoampere","it_IT":"Hectoampere","ja_JP":"ヘクトアンペア","pt_BR":"Hectoampere","ru_RU":"Гектоампер","sv_SE":"Hectoampere"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"hA"},{"code":"KILOAMPERE","labels":{"ca_ES":"Quiloampere","da_DK":"Kiloampere","de_DE":"Kiloampere","en_GB":"Kiloampere","en_NZ":"Kiloampere","en_US":"Kiloampere","es_ES":"Kiloamperio","fi_FI":"Kiloampeeri","fr_FR":"Kiloampère","it_IT":"Kiloampere","ja_JP":"キロアンペア","pt_BR":"Kiloampere","ru_RU":"Килоампер","sv_SE":"Kiloampere"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kA"}]'),
    ('Resistance', '{"ca_ES":"Resistència","da_DK":"Modstand","de_DE":"Widerstand","en_GB":"Resistance","en_NZ":"Resistence","en_US":"Resistence","es_ES":"Resistencia","fi_FI":"Resistanssi","fr_FR":"Résistance","it_IT":"Resistenza","ja_JP":"抵抗","pt_BR":"Resistência","ru_RU":"Сопротивление","sv_SE":"Motstånd"}', 'OHM', '[{"code":"MILLIOHM","labels":{"ca_ES":"Mil·liohm","da_DK":"Milliohm","de_DE":"Milliohm","en_GB":"Milliohm","en_NZ":"Milliohm","en_US":"Milliohm","es_ES":"Miliohmio","fi_FI":"Milliohmi","fr_FR":"Milliohm","it_IT":"Milliohm","ja_JP":"ミリオーム","pt_BR":"Milliohm","ru_RU":"Миллиом","sv_SE":"Milliohm"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mΩ"},{"code":"CENTIOHM","labels":{"ca_ES":"Centiohm","da_DK":"Centiohm","de_DE":"Centiohm","en_GB":"Centiohm","en_NZ":"Centiohm","en_US":"Centiohm","es_ES":"Centiohmio","fi_FI":"Senttiohmi","fr_FR":"Centiohm","it_IT":"Centiohm","ja_JP":"センチオーム","pt_BR":"Centiohm","ru_RU":"Сантиом","sv_SE":"Centiohm"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"cΩ"},{"code":"DECIOHM","labels":{"ca_ES":"Deciohm","da_DK":"Deciohm","de_DE":"Deciohm","en_GB":"Deciohm","en_NZ":"Deciohm","en_US":"Deciohm","es_ES":"Deciohmio","fi_FI":"Desiohmi","fr_FR":"Déciohm","it_IT":"Deciohm","ja_JP":"デシオーム","pt_BR":"Deciohm","ru_RU":"Дециом","sv_SE":"Deciohm"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"dΩ"},{"code":"OHM","labels":{"ca_ES":"Ohm","da_DK":"Ohm","de_DE":"Ohm","en_GB":"Ohm","en_NZ":"Ohm","en_US":"Ohm","es_ES":"Ohmio","fi_FI":"Ohmi","fr_FR":"Ohm","it_IT":"Ohm","ja_JP":"オーム","pt_BR":"Ohm","ru_RU":"Ом","sv_SE":"Ohm"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"Ω"},{"code":"DEKAOHM","labels":{"ca_ES":"Decaohm","da_DK":"Dekaohm","de_DE":"Dekaohm","en_GB":"Decaohm","en_NZ":"Dekaohm","en_US":"Dekaohm","es_ES":"Decaohmio","fi_FI":"Dekaohmi","fr_FR":"Decaohm","it_IT":"Decaohm","ja_JP":"デカオーム","pt_BR":"Dekaohm","ru_RU":"Декаом","sv_SE":"Dekaohm"},"convert_from_standard":[{"operator":"mul","value":"10"}],"symbol":"daΩ"},{"code":"HECTOHM","labels":{"ca_ES":"Hectohm","da_DK":"Hectohm","de_DE":"Hectohm","en_GB":"Hectohm","en_NZ":"Hectohm","en_US":"Hectohm","es_ES":"Hectoohmio","fi_FI":"Hehto-ohmi","fr_FR":"Hectohm","it_IT":"Hectohm","ja_JP":"ヘクトオーム","pt_BR":"Hectohm","ru_RU":"Гектоом","sv_SE":"Hectaohm"},"convert_from_standard":[{"operator":"mul","value":"100"}],"symbol":"hΩ"},{"code":"KILOHM","labels":{"ca_ES":"Quilohm","da_DK":"Kilohm","de_DE":"KOhm","en_GB":"Kilohm","en_NZ":"Kilohm","en_US":"Kilohm","es_ES":"Kiloohmio","fi_FI":"Kilo-ohmi","fr_FR":"Kilohm","it_IT":"Kiloohm","ja_JP":"キロオーム","pt_BR":"Kilohm","ru_RU":"Килоом","sv_SE":"Kilohm"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kΩ"},{"code":"MEGOHM","labels":{"ca_ES":"Megohm","da_DK":"Megohm","de_DE":"Megohm","en_GB":"Megohm","en_NZ":"Megohm","en_US":"Megohm","es_ES":"Megaohmio","fi_FI":"Megaohmi","fr_FR":"Mégohm","it_IT":"Megaohm","ja_JP":"メグオーム","pt_BR":"Megohm","ru_RU":"Мегаом","sv_SE":"Megaohm"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"MΩ"}]'),
    ('Speed', '{"ca_ES":"Velocitat","da_DK":"Hastighed","de_DE":"Geschwindigkeit","en_GB":"Speed","en_NZ":"Speed","en_US":"Speed","es_ES":"Velocidad","fi_FI":"Nopeus","fr_FR":"Vitesse","it_IT":"Velocità","ja_JP":"速度","pt_BR":"Velocidade","ru_RU":"Скорость","sv_SE":"Hastighet"}', 'METER_PER_SECOND', '[{"code":"METER_PER_SECOND","labels":{"ca_ES":"Metre per segon","da_DK":"Meter pr. sekund","de_DE":"Meter pro Sekunde","en_GB":"Metre per second","en_NZ":"Metre per second","en_US":"Meter per second","es_ES":"Metro por segundo","fi_FI":"Metriä sekunnissa","fr_FR":"Mètre par seconde","it_IT":"Metri al secondo","ja_JP":"m/秒","pt_BR":"Metro por segundo","ru_RU":"Метров в секунду","sv_SE":"Meter per sekund"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"m/s"},{"code":"METER_PER_MINUTE","labels":{"ca_ES":"Metre per minut","da_DK":"Meter pr. minut","de_DE":"Meter pro minute","en_GB":"Metre per minute","en_NZ":"Metre per minute","en_US":"Meter per minute","es_ES":"Metro por minuto","fi_FI":"Metriä minuutissa","fr_FR":"Mètre par minute","it_IT":"Metri al minuto","ja_JP":"m/分","pt_BR":"Metro por minuto","ru_RU":"Метров в минуту","sv_SE":"Meter per minut"},"convert_from_standard":[{"operator":"div","value":"60"}],"symbol":"m/mn"},{"code":"METER_PER_HOUR","labels":{"ca_ES":"Metre per hora","da_DK":"Meter i timen","de_DE":"Meter pro Stunde","en_GB":"Metre per hour","en_NZ":"Metre per hour","en_US":"Meter per hour","es_ES":"Metro por hora","fi_FI":"Metriä tunnissa","fr_FR":"Mètre par heure","it_IT":"Metri all\'ora","ja_JP":"m/時","pt_BR":"Metro por hora","ru_RU":"Метров в час","sv_SE":"Meter per timme"},"convert_from_standard":[{"operator":"mul","value":"1"},{"operator":"div","value":"3600"}],"symbol":"m/h"},{"code":"KILOMETER_PER_HOUR","labels":{"ca_ES":"Quilòmetre per hora","da_DK":"Kilometer i timen","de_DE":"Kilometer pro Stunde","en_GB":"Kilometre per hour","en_NZ":"Kilometre per hour","en_US":"Kilometer per hour","es_ES":"Kilómetro por hora","fi_FI":"Kilometriä tunnissa","fr_FR":"Kilomètre par heure","it_IT":"Chilometri all\'ora","ja_JP":"Km/時","pt_BR":"Quilômetro por hora","ru_RU":"Километров в час","sv_SE":"Kilometer per timme"},"convert_from_standard":[{"operator":"mul","value":"1000"},{"operator":"div","value":"3600"}],"symbol":"km/h"},{"code":"FOOT_PER_SECOND","labels":{"ca_ES":"Peu per segon","da_DK":"Fod pr. sekund","de_DE":"Fuß pro Sekunde","en_GB":"Foot per second","en_NZ":"Foot per second","en_US":"Foot per second","es_ES":"Pie por segundo","fi_FI":"Jalkaa sekunnissa","fr_FR":"Pied par seconde","it_IT":"Piedi al secondo","ja_JP":"フィート/秒","pt_BR":"Pé por segundo","ru_RU":"Футов в секунду","sv_SE":"Fot per sekund"},"convert_from_standard":[{"operator":"mul","value":"0.3048"}],"symbol":"ft/s"},{"code":"FOOT_PER_HOUR","labels":{"ca_ES":"Peu per hora","da_DK":"Fod pr. time","de_DE":"Fuß pro Stunde","en_GB":"Foot per hour","en_NZ":"Foot per hour","en_US":"Foot per hour","es_ES":"Pie por hora","fi_FI":"Jalkaa tunnissa","fr_FR":"Pied / heure","it_IT":"Piedi all\'ora","ja_JP":"フィート/時","pt_BR":"Pé por hora","ru_RU":"Футoв в час","sv_SE":"Fot per timme"},"convert_from_standard":[{"operator":"mul","value":"0.3048"},{"operator":"div","value":"3600"}],"symbol":"ft/h"},{"code":"YARD_PER_HOUR","labels":{"ca_ES":"Iarda per hora","da_DK":"Yard i timen","de_DE":"Yard pro Stunde","en_GB":"Yard per hour","en_NZ":"Yard per hour","en_US":"Yard per hour","es_ES":"Yarda por hora","fi_FI":"Jaardia tunnissa","fr_FR":"Yard par heure","it_IT":"Yard orarie","ja_JP":"ヤード/時","pt_BR":"Jarda por hora","ru_RU":"Ярдов в час","sv_SE":"Yard per timme"},"convert_from_standard":[{"operator":"mul","value":"0.9144"},{"operator":"div","value":"3600"}],"symbol":"yd/h"},{"code":"MILE_PER_HOUR","labels":{"ca_ES":"Milla per hora","da_DK":"Mile i timen","de_DE":"Meilen pro Stunde","en_GB":"Mile per hour","en_NZ":"Mile per hour","en_US":"Mile per hour","es_ES":"Millas por hora","fi_FI":"Mailia tunnissa","fr_FR":"Mile par heure","it_IT":"Miglia orarie","ja_JP":"マイル/時","pt_BR":"Milha por hora","ru_RU":"Миль в час","sv_SE":"Mile per timme"},"convert_from_standard":[{"operator":"mul","value":"1609.344"},{"operator":"div","value":"3600"}],"symbol":"mi/h"}]'),
    ('ElectricCharge', '{"ca_ES":"Càrrega elèctrica","da_DK":"Elektrisk ladning","de_DE":"Elektrische Ladung","en_GB":"Electrical charge","en_NZ":"Electric charge","en_US":"Electric charge","es_ES":"Carga eléctrica","fi_FI":"Sähkövaraus","fr_FR":"Charge électrique","it_IT":"Carica elettrica","ja_JP":"電気料金","pt_BR":"Carga elétrica","ru_RU":"Электрический заряд","sv_SE":"Elektrisk laddning"}', 'AMPEREHOUR', '[{"code":"MILLIAMPEREHOUR","labels":{"ca_ES":"Microampers per hora","da_DK":"Milliampere time","de_DE":"Milliampere-Stunden","en_GB":"Milliampere hour","en_NZ":"Milliampere hour","en_US":"Milliampere hour","es_ES":"Miliamperio por hora","fi_FI":"Milliampeeritunti","fr_FR":"Milliampères heure","it_IT":"Milliamperora","ja_JP":"ミリ アンペア時間","pt_BR":"Miliampère/hora","ru_RU":"Миллиампер в час","sv_SE":"Milliampere timme"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mAh"},{"code":"AMPEREHOUR","labels":{"ca_ES":"Ampers per hora","da_DK":"Ampere timer","de_DE":"Ampere-Stunden","en_GB":"Ampere hour","en_NZ":"Ampere hour","en_US":"Ampere hour","es_ES":"Amperio por hora","fi_FI":"Ampeeritunti","fr_FR":"Ampère heure","it_IT":"Amperora","ja_JP":"アンペア時間","pt_BR":"Ampère/hora","ru_RU":"Ампер в час","sv_SE":"Ampere timmar"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"Ah"},{"code":"MILLICOULOMB","labels":{"ca_ES":"Mil·licoulomb","da_DK":"Millicoulomb","de_DE":"Millicoulomb","en_GB":"Millicoulomb","en_NZ":"Millicoulomb","en_US":"Millicoulomb","es_ES":"Miliculombio","fi_FI":"Millicoulombi","fr_FR":"Millicoulomb","it_IT":"Millicoulomb","ja_JP":"ミリクーロン","pt_BR":"Milicoulomb","ru_RU":"Милликулон","sv_SE":"Millicoulomb"},"convert_from_standard":[{"operator":"div","value":"3600000"}],"symbol":"mC"},{"code":"CENTICOULOMB","labels":{"ca_ES":"Centicoulomb","da_DK":"Centicoulomb","de_DE":"Centicoulomb","en_GB":"Centicoulomb","en_NZ":"Centicoulomb","en_US":"Centicoulomb","es_ES":"Centiculombio","fi_FI":"Sentticoulombi","fr_FR":"Centicoulomb","it_IT":"Centicoulomb","ja_JP":"センチクーロン","pt_BR":"Centicoulomb","ru_RU":"Сантикулон","sv_SE":"Centicoulomb"},"convert_from_standard":[{"operator":"div","value":"360000"}],"symbol":"cC"},{"code":"DECICOULOMB","labels":{"ca_ES":"Decicoulomb","da_DK":"Decicoulomb","de_DE":"Decicoulomb","en_GB":"Decicoulomb","en_NZ":"Decicoulomb","en_US":"Decicoulomb","es_ES":"Deciculombio","fi_FI":"Desicoulombi","fr_FR":"Décicoulomb","it_IT":"Decicoulomb","ja_JP":"デシクーロン","pt_BR":"Decicoulomb","ru_RU":"Децикулон","sv_SE":"Decicoulomb"},"convert_from_standard":[{"operator":"div","value":"36000"}],"symbol":"dC"},{"code":"COULOMB","labels":{"ca_ES":"Coulomb","da_DK":"Coulomb","de_DE":"Coulomb","en_GB":"Coulomb","en_NZ":"Coulomb","en_US":"Coulomb","es_ES":"Culombio","fi_FI":"Coulombi","fr_FR":"Coulomb","it_IT":"Coulomb","ja_JP":"クーロン","pt_BR":"Coulomb","ru_RU":"Кулон","sv_SE":"Coulomb"},"convert_from_standard":[{"operator":"div","value":"3600"}],"symbol":"C"},{"code":"DEKACOULOMB","labels":{"ca_ES":"Decacoulomb","da_DK":"Dekacoulomb","de_DE":"Dekacoulomb","en_GB":"Dekacoulomb","en_NZ":"Dekacoulomb","en_US":"Dekacoulomb","es_ES":"Decaculombio","fi_FI":"Dekacoulombi","fr_FR":"Décacoulomb","it_IT":"Decacoulomb","ja_JP":"デカクーロン","pt_BR":"Dekacoulomb","ru_RU":"Декакулон","sv_SE":"Dekacoulomb"},"convert_from_standard":[{"operator":"div","value":"360"}],"symbol":"daC"},{"code":"HECTOCOULOMB","labels":{"ca_ES":"Hectocoulomb","da_DK":"Hectocoulomb","de_DE":"Hectocoulomb","en_GB":"Hectocoulomb","en_NZ":"Hectocoulomb","en_US":"Hectocoulomb","es_ES":"Hectoculombio","fi_FI":"Hehtocoulombi","fr_FR":"Hectocoulomb","it_IT":"Hectocoulomb","ja_JP":"ヘクトクーロン","pt_BR":"Hectocoulomb","ru_RU":"Гектокулон","sv_SE":"Hectocoulomb"},"convert_from_standard":[{"operator":"div","value":"36"}],"symbol":"hC"},{"code":"KILOCOULOMB","labels":{"ca_ES":"Quilocoulomb","da_DK":"Kilocoulomb","de_DE":"Kilocoulomb","en_GB":"Kilocoulomb","en_NZ":"Kilocoulomb","en_US":"Kilocoulomb","es_ES":"Kiloculombio","fi_FI":"Kilocoulombi","fr_FR":"Kilocoulomb","it_IT":"Kilocoulomb","ja_JP":"キロクーロン","pt_BR":"Quilocoulomb","ru_RU":"Килокулон","sv_SE":"Kilocoulomb"},"convert_from_standard":[{"operator":"div","value":"3.6"}],"symbol":"kC"}]'),
    ('Duration', '{"ca_ES":"Durada","da_DK":"Varighed","de_DE":"Dauer","en_GB":"Duration","en_NZ":"Duration","en_US":"Duration","es_ES":"Duración","fi_FI":"Kesto","fr_FR":"Durée","it_IT":"Durata","ja_JP":"期間","pt_BR":"Duração","ru_RU":"Длительности","sv_SE":"Varaktighet"}', 'SECOND', '[{"code":"MILLISECOND","labels":{"ca_ES":"Mil·lisegon","da_DK":"Millisekund","de_DE":"Millisekunde","en_GB":"Millisecond","en_NZ":"Millisecond","en_US":"Millisecond","es_ES":"Milisegundo","fi_FI":"Millisekunti","fr_FR":"Milliseconde","it_IT":"Millisecondi","ja_JP":"ミリ秒","pt_BR":"Milissegundo","ru_RU":"Миллисекунда","sv_SE":"Millisekund"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"ms"},{"code":"SECOND","labels":{"ca_ES":"Segon","da_DK":"Sekund","de_DE":"Sekunde","en_GB":"Second","en_NZ":"Second","en_US":"Second","es_ES":"Segundo","fi_FI":"Sekunti","fr_FR":"Seconde","it_IT":"Secondi","ja_JP":"秒","pt_BR":"Segundo","ru_RU":"Секунда","sv_SE":"Sekund"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"s"},{"code":"MINUTE","labels":{"ca_ES":"Minut","da_DK":"Minut","de_DE":"Minute","en_GB":"Minute","en_NZ":"Minute","en_US":"Minute","es_ES":"Minuto","fi_FI":"Minuutti","fr_FR":"Minute","it_IT":"Minuti","ja_JP":"分","pt_BR":"Minuto","ru_RU":"Минута","sv_SE":"Minut"},"convert_from_standard":[{"operator":"mul","value":"60"}],"symbol":"m"},{"code":"HOUR","labels":{"ca_ES":"Hora","da_DK":"Time","de_DE":"Stunde","en_GB":"Hour","en_NZ":"Hour","en_US":"Hour","es_ES":"Hora","fi_FI":"Tunti","fr_FR":"Heure","it_IT":"Ora","ja_JP":"時","pt_BR":"Hora","ru_RU":"Час","sv_SE":"Timme"},"convert_from_standard":[{"operator":"mul","value":"3600"}],"symbol":"h"},{"code":"DAY","labels":{"ca_ES":"Dia","da_DK":"Dag","de_DE":"Tag","en_GB":"Day","en_NZ":"Day","en_US":"Day","es_ES":"Día","fi_FI":"Päivä","fr_FR":"Jour","it_IT":"Giorno","ja_JP":"日","pt_BR":"Dia","ru_RU":"День","sv_SE":"Dag"},"convert_from_standard":[{"operator":"mul","value":"86400"}],"symbol":"d"},{"code":"WEEK","labels":{"ca_ES":"Setmana","da_DK":"Uge","de_DE":"Woche","en_GB":"Week","en_NZ":"Week","en_US":"Week","es_ES":"Semana","fi_FI":"Viikko","fr_FR":"Semaine","it_IT":"Settimana","ja_JP":"週","pt_BR":"Semana","ru_RU":"Неделя","sv_SE":"Vecka"},"convert_from_standard":[{"operator":"mul","value":"604800"}],"symbol":"week"},{"code":"MONTH","labels":{"ca_ES":"Mes","da_DK":"Måned","de_DE":"Monat","en_GB":"Month","en_NZ":"Month","en_US":"Month","es_ES":"Mes","fi_FI":"Kuukausi","fr_FR":"Mois","it_IT":"Mese","ja_JP":" 月","pt_BR":"Mês","ru_RU":"месяцев","sv_SE":"Månad"},"convert_from_standard":[{"operator":"mul","value":"2628000"}],"symbol":"month"},{"code":"YEAR","labels":{"ca_ES":"Any","da_DK":"År","de_DE":"Jahr","en_GB":"Year","en_NZ":"Year","en_US":"Year","es_ES":"Año","fi_FI":"Vuosi","fr_FR":"Année","it_IT":"Anno","ja_JP":"年","pt_BR":"Ano","ru_RU":"лет","sv_SE":"År"},"convert_from_standard":[{"operator":"mul","value":"31536000"}],"symbol":"year"}]'),
    ('Temperature', '{"ca_ES":"Temperatura","da_DK":"Temperatur","de_DE":"Temperatur","en_GB":"Temperature","en_NZ":"Temperature","en_US":"Temperature","es_ES":"Temperatura","fi_FI":"Lämpötila","fr_FR":"Température","it_IT":"Temperatura","ja_JP":"温度","pt_BR":"Temperatura","ru_RU":"Температура","sv_SE":"Temperatur"}', 'KELVIN', '[{"code":"CELSIUS","labels":{"ca_ES":"Celsius","da_DK":"Celsius","de_DE":"Celsius","en_GB":"Celsius","en_NZ":"Celsius","en_US":"Celsius","es_ES":"Celsius","fi_FI":"Celsius","fr_FR":"Celsius","it_IT":"Celsius","ja_JP":"摂氏","pt_BR":"Celsius","ru_RU":"По Цельсию","sv_SE":"Celsius"},"convert_from_standard":[{"operator":"add","value":"273.15"}],"symbol":"°C"},{"code":"FAHRENHEIT","labels":{"ca_ES":"Fahrenheit","da_DK":"Fahrenheit","de_DE":"Fahrenheit","en_GB":"Fahrenheit","en_NZ":"Fahrenheit","en_US":"Fahrenheit","es_ES":"Fahrenheit","fi_FI":"Fahrenheit","fr_FR":"Fahrenheit","it_IT":"Fahrenheit","ja_JP":"華氏","pt_BR":"Fahrenheit","ru_RU":"По Фаренгейту","sv_SE":"Fahrenheit"},"convert_from_standard":[{"operator":"sub","value":"32"},{"operator":"div","value":"1.8"},{"operator":"add","value":"273.15"}],"symbol":"°F"},{"code":"KELVIN","labels":{"ca_ES":"Kelvin","da_DK":"Kelvin","de_DE":"Kelvin","en_GB":"Kelvin","en_NZ":"Kelvin","en_US":"Kelvin","es_ES":"Kelvin","fi_FI":"Kelvin","fr_FR":"Kelvin","it_IT":"Kelvin","ja_JP":"ケルビン","pt_BR":"Kelvin","ru_RU":"По Кельвину","sv_SE":"Kelvin"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"°K"},{"code":"RANKINE","labels":{"ca_ES":"Rankine","da_DK":"Rankine","de_DE":"Rankine","en_GB":"Rankine","en_NZ":"Rankine","en_US":"Rankine","es_ES":"Rankine","fi_FI":"Rankine","fr_FR":"Rankine","it_IT":"Rankine","ja_JP":"ランキン","pt_BR":"Rankine","ru_RU":"По Ранкину","sv_SE":"Rankine"},"convert_from_standard":[{"operator":"div","value":"1.8"}],"symbol":"°R"},{"code":"REAUMUR","labels":{"ca_ES":"Reaumur","da_DK":"Reaumur","de_DE":"Réaumur","en_GB":"Réaumur","en_NZ":"Reaumur","en_US":"Reaumur","es_ES":"Reaumur","fi_FI":"Reaumur","fr_FR":"Réaumur","it_IT":"Reaumur","ja_JP":"列氏","pt_BR":"Reaumur","ru_RU":"По Реомюру","sv_SE":"Reaumur"},"convert_from_standard":[{"operator":"mul","value":"1.25"},{"operator":"add","value":"273.15"}],"symbol":"°r"}]'),
    ('Volume', '{"ca_ES":"Volum","da_DK":"Volume","de_DE":"Volumen","en_GB":"Volume","en_NZ":"Volume","en_US":"Volume","es_ES":"Volumen","fi_FI":"Tilavuus","fr_FR":"Volume","it_IT":"Volume","ja_JP":"ボリューム","pt_BR":"Volume","ru_RU":"Объем","sv_SE":"Volym"}', 'CUBIC_METER', '[{"code":"MICROLITER","labels":{"fr_FR":"Microlitre","en_US":"Microliter"},"convert_from_standard":[{"operator":"mul","value":"0.000000001"}],"symbol":"μl"},{"code":"HECTOLITER","labels":{"fr_FR":"Hectolitre","en_US":"Hectoliter"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"hl"},{"code":"CUBIC_MILLIMETER","labels":{"ca_ES":"Mil·límetre cúbic","da_DK":"Kubikmillimeter","de_DE":"Kubikmillimeter","en_GB":"Cubic millimetre","en_NZ":"Cubic millimetre","en_US":"Cubic millimeter","es_ES":"Milímetro cúbico","fi_FI":"Kuutiomillimetri","fr_FR":"Millimètre cube","it_IT":"Millimetro cubo","ja_JP":"立方ミリメートル","pt_BR":"Milímetro cúbico","ru_RU":"Кубический миллиметр","sv_SE":"Kubikmillimeter"},"convert_from_standard":[{"operator":"mul","value":"0.000000001"}],"symbol":"mm³"},{"code":"CUBIC_CENTIMETER","labels":{"ca_ES":"Centímetre cúbic","da_DK":"Kubikcentimeter","de_DE":"Kubikzentimeter","en_GB":"Cubic centimetre","en_NZ":"Cubic centimetre","en_US":"Cubic centimeter","es_ES":"Centímetro cúbico","fi_FI":"Kuutiosenttimetri","fr_FR":"Centimètre cube","it_IT":"Centimetro cubo","ja_JP":"立方センチメートル","pt_BR":"Centímetro cúbico","ru_RU":"Кубический сантиметр","sv_SE":"Kubikcentimeter"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"cm³"},{"code":"MILLILITER","labels":{"ca_ES":"Mililitre","da_DK":"Milliliter","de_DE":"Milliliter","en_GB":"Millilitre","en_NZ":"Millilitre","en_US":"Milliliter","es_ES":"Mililitro","fi_FI":"Millilitra","fr_FR":"Millilitre","it_IT":"Millilitro","ja_JP":"ミリリットル","pt_BR":"Mililitro","ru_RU":"Миллилитр","sv_SE":"Milliliter"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"ml"},{"code":"CENTILITER","labels":{"ca_ES":"Centilitre","da_DK":"Centiliter","de_DE":"Zentiliter","en_GB":"Centilitre","en_NZ":"Centilitre","en_US":"Centiliter","es_ES":"Centilítro","fi_FI":"Senttilitra","fr_FR":"Centilitre","it_IT":"Centilitri","ja_JP":"センチリットル","pt_BR":"Centilitro","ru_RU":"Сантилитр","sv_SE":"Centiliter"},"convert_from_standard":[{"operator":"mul","value":"0.00001"}],"symbol":"cl"},{"code":"DECILITER","labels":{"ca_ES":"Decilitre","da_DK":"Deciliter","de_DE":"Deziliter","en_GB":"Decilitre","en_NZ":"Decilitre","en_US":"Deciliter","es_ES":"Decilitro","fi_FI":"Desilitra","fr_FR":"Décilitre","it_IT":"Decilitro","ja_JP":"デシリットル","pt_BR":"Decilitro","ru_RU":"Децилитр","sv_SE":"Deciliter"},"convert_from_standard":[{"operator":"mul","value":"0.0001"}],"symbol":"dl"},{"code":"CUBIC_DECIMETER","labels":{"ca_ES":"Decímetre cúbic","da_DK":"Kubik decimeter","de_DE":"Kubikdezimeter","en_GB":"Cubic decimetre","en_NZ":"Cubic decimetre","en_US":"Cubic decimeter","es_ES":"Decímetro cúbico","fi_FI":"Kuutiodesimetri","fr_FR":"Décimètre cube","it_IT":"Decimetro cubo","ja_JP":"立方デシメートル","pt_BR":"Decímetro cúbico","ru_RU":"Кубический дециметр","sv_SE":"Kubikdecimeter"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"dm³"},{"code":"LITER","labels":{"ca_ES":"Litre","da_DK":"Liter","de_DE":"Liter","en_GB":"Litre","en_NZ":"Litre","en_US":"Liter","es_ES":"Litro","fi_FI":"Litra","fr_FR":"Litre","it_IT":"Litri","ja_JP":"リットル","pt_BR":"Litro","ru_RU":"Литр","sv_SE":"Liter"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"l"},{"code":"CUBIC_METER","labels":{"ca_ES":"Metre cúbic","da_DK":"Kubikmeter","de_DE":"Kubikmeter","en_GB":"Cubic metre","en_NZ":"Cubic metre","en_US":"Cubic meter","es_ES":"Metro cúbico","fi_FI":"Kuutiometri","fr_FR":"Mètre cube","it_IT":"Metro cubo","ja_JP":"立方メートル","pt_BR":"Metro cúbico","ru_RU":"Кубический метр","sv_SE":"Kubikmeter"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"m³"},{"code":"OUNCE","labels":{"ca_ES":"Unça","da_DK":"Ounce","de_DE":"Unze","en_GB":"Ounce","en_NZ":"Ounce","en_US":"Ounce","es_ES":"Onza","fi_FI":"Unssi","fr_FR":"Once","it_IT":"Oncia","ja_JP":"オンス","pt_BR":"Onça","ru_RU":"Унция","sv_SE":"Ounce"},"convert_from_standard":[{"operator":"mul","value":"0.00454609"},{"operator":"div","value":"160"}],"symbol":"oz"},{"code":"PINT","labels":{"ca_ES":"Pinta","da_DK":"Pint","de_DE":"Pint","en_GB":"Pint","en_NZ":"Pint","en_US":"Pint","es_ES":"Pinta","fi_FI":"Pintti","fr_FR":"Pinte","it_IT":"Pinta","ja_JP":"パイント","pt_BR":"Pinta","ru_RU":"Пинта","sv_SE":"Pint"},"convert_from_standard":[{"operator":"mul","value":"0.00454609"},{"operator":"div","value":"8"}],"symbol":"pt"},{"code":"BARREL","labels":{"ca_ES":"Barril","da_DK":"Tønde","de_DE":"Barrel","en_GB":"Barrel","en_NZ":"Barrel","en_US":"Barrel","es_ES":"Barril","fi_FI":"Tynnyri","fr_FR":"Baril","it_IT":"Barrel","ja_JP":"バレル","pt_BR":"Barril","ru_RU":"Баррель","sv_SE":"Fat"},"convert_from_standard":[{"operator":"mul","value":"0.16365924"}],"symbol":"bbl"},{"code":"GALLON","labels":{"ca_ES":"Galó","da_DK":"Gallon","de_DE":"Gallone","en_GB":"Gallon","en_NZ":"Gallon","en_US":"Gallon","es_ES":"Galón","fi_FI":"Gallona","fr_FR":"Gallon","it_IT":"Gallone","ja_JP":"ガロン","pt_BR":"Galão","ru_RU":"Галлон","sv_SE":"Gallon"},"convert_from_standard":[{"operator":"mul","value":"0.00454609"}],"symbol":"gal"},{"code":"CUBIC_FOOT","labels":{"ca_ES":"Peu cúbic","da_DK":"Kubikfod","de_DE":"Kubikfuß","en_GB":"Cubic foot","en_NZ":"Cubic foot","en_US":"Cubic foot","es_ES":"Pie cúbico","fi_FI":"Kuutiojalka","fr_FR":"Pied cube","it_IT":"Piede cubo","ja_JP":"立方フィート","pt_BR":"Pé cúbico","ru_RU":"Кубический фут","sv_SE":"Kubikfot"},"convert_from_standard":[{"operator":"mul","value":"6.54119159"},{"operator":"div","value":"231"}],"symbol":"ft³"},{"code":"CUBIC_INCH","labels":{"ca_ES":"Polzada cúbica","da_DK":"Kubik tommer","de_DE":"Kubikzoll","en_GB":"Cubic inch","en_NZ":"Cubic inch","en_US":"Cubic inch","es_ES":"Pulgada cúbica","fi_FI":"Kuutiotuuma","fr_FR":"Pouce cube","it_IT":"Pollice cubo","ja_JP":"立方インチ","pt_BR":"Polegada cúbica","ru_RU":"Кубический дюйм","sv_SE":"Kubiktum"},"convert_from_standard":[{"operator":"mul","value":"0.0037854118"},{"operator":"div","value":"231"}],"symbol":"in³"},{"code":"CUBIC_YARD","labels":{"ca_ES":"Iarda cúbica","da_DK":"Kubik yard","de_DE":"Kubikyard","en_GB":"Cubic yard","en_NZ":"Cubic yard","en_US":"Cubic yard","es_ES":"Yarda cúbica","fi_FI":"Kuutiojaardi","fr_FR":"Yard cube","it_IT":"Iarda cubica","ja_JP":"立方ヤード","pt_BR":"Jarda cúbica","ru_RU":"Кубический ярд","sv_SE":"Kubikyard"},"convert_from_standard":[{"operator":"mul","value":"0.764554861"}],"symbol":"yd³"}]'),
    ('Weight', '{"ca_ES":"Pes","da_DK":"Vægt","de_DE":"Gewicht","en_GB":"Weight","en_NZ":"Weight","en_US":"Weight","es_ES":"Peso","fi_FI":"Paino","fr_FR":"Poids","it_IT":"Peso","ja_JP":"重量","pt_BR":"Peso","ru_RU":"Вес","sv_SE":"Vikt"}', 'KILOGRAM', '[{"code":"MICROGRAM","labels":{"fr_FR":"Microgramme","en_US":"Microgram"},"convert_from_standard":[{"operator":"mul","value":"0.000000001"}],"symbol":"μg"},{"code":"MILLIGRAM","labels":{"ca_ES":"Mil·ligram","da_DK":"Milligram","de_DE":"Milligramm","en_GB":"Milligram","en_NZ":"Milligram","en_US":"Milligram","es_ES":"Miligramo","fi_FI":"Milligramma","fr_FR":"Milligramme","it_IT":"Milligrammo","ja_JP":"ミリグラム","pt_BR":"Miligrama","ru_RU":"Миллиграмм","sv_SE":"Milligram"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"mg"},{"code":"GRAM","labels":{"ca_ES":"Gram","da_DK":"Gram","de_DE":"Gramm","en_GB":"Gram","en_NZ":"Gram","en_US":"Gram","es_ES":"Gramo","fi_FI":"Gramma","fr_FR":"Gramme","it_IT":"Gram","ja_JP":"グラム","pt_BR":"Grama","ru_RU":"Грамм","sv_SE":"Gram"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"g"},{"code":"KILOGRAM","labels":{"ca_ES":"Quilogram","da_DK":"Kilogram","de_DE":"Kilogramm","en_GB":"Kilogram","en_NZ":"Kilogram","en_US":"Kilogram","es_ES":"Kilogramos","fi_FI":"Kilogramma","fr_FR":"Kilogramme","it_IT":"Chilogrammo","ja_JP":"キログラム","pt_BR":"Quilograma","ru_RU":"Килограмм","sv_SE":"Kilogram"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"kg"},{"code":"TON","labels":{"ca_ES":"Tona","da_DK":"Ton","de_DE":"Tonne","en_GB":"Ton","en_NZ":"Ton","en_US":"Ton","es_ES":"Tonelada","fi_FI":"Tonni","fr_FR":"Tonne","it_IT":"Tonnellata","ja_JP":"トン","pt_BR":"Tonelada","ru_RU":"Тонна","sv_SE":"Ton"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"t"},{"code":"GRAIN","labels":{"ca_ES":"Gra","da_DK":"Gran","de_DE":"Korn","en_GB":"Grain","en_NZ":"Grain","en_US":"Grain","es_ES":"Granulado","fi_FI":"Graani","fr_FR":"Grain","it_IT":"Grain","ja_JP":"粒","pt_BR":"Grão","ru_RU":"Гран","sv_SE":"Grain"},"convert_from_standard":[{"operator":"mul","value":"0.00006479891"}],"symbol":"gr"},{"code":"DENIER","labels":{"ca_ES":"Denier","da_DK":"Denier","de_DE":"Denier","en_GB":"Denier","en_NZ":"Denier","en_US":"Denier","es_ES":"Denier","fi_FI":"Denier","fr_FR":"Denier","it_IT":"Denier","ja_JP":"デニール","pt_BR":"Denier","ru_RU":"Денье","sv_SE":"Denier"},"convert_from_standard":[{"operator":"mul","value":"0.001275"}],"symbol":"denier"},{"code":"ONCE","labels":{"ca_ES":"Unça","da_DK":"Ounce","de_DE":"Unze","en_GB":"Once","en_NZ":"Once","en_US":"Once","es_ES":"Onza","fi_FI":"Unssi","fr_FR":"Once française","it_IT":"Once","ja_JP":"オンス","pt_BR":"Onça","ru_RU":"Унция","sv_SE":"Once"},"convert_from_standard":[{"operator":"mul","value":"0.03059"}],"symbol":"once"},{"code":"MARC","labels":{"ca_ES":"Marc","da_DK":"Marc","de_DE":"Mark","en_GB":"Marc","en_NZ":"Marc","en_US":"Marc","es_ES":"Marc","fi_FI":"Marc","fr_FR":"Marc","it_IT":"Marc","ja_JP":"マルク","pt_BR":"Marc","ru_RU":"Марка","sv_SE":"Marc"},"convert_from_standard":[{"operator":"mul","value":"0.24475"}],"symbol":"marc"},{"code":"LIVRE","labels":{"ca_ES":"Lliura","da_DK":"Pund","de_DE":"Livre","en_GB":"Livre","en_NZ":"Livre","en_US":"Livre","es_ES":"Libra","fi_FI":"Livre","fr_FR":"Livre française","it_IT":"Livre","ja_JP":"リーブル","pt_BR":"Livre","ru_RU":"Фунт","sv_SE":"Livre"},"convert_from_standard":[{"operator":"mul","value":"0.4895"}],"symbol":"livre"},{"code":"OUNCE","labels":{"ca_ES":"Unça","da_DK":"Ounce","de_DE":"Unze","en_GB":"Ounce","en_NZ":"Ounce","en_US":"Ounce","es_ES":"Onza","fi_FI":"Unssi","fr_FR":"Once","it_IT":"Oncia","ja_JP":"オンス","pt_BR":"Onça","ru_RU":"Унция","sv_SE":"Ounce"},"convert_from_standard":[{"operator":"mul","value":"0.45359237"},{"operator":"div","value":"16"}],"symbol":"oz"},{"code":"POUND","labels":{"ca_ES":"Lliura","da_DK":"Pund","de_DE":"Pfund","en_GB":"Pound","en_NZ":"Pound","en_US":"Pound","es_ES":"Libra","fi_FI":"Pauna","fr_FR":"Livre","it_IT":"Pound","ja_JP":"ポンド","pt_BR":"Libra","ru_RU":"Фунт","sv_SE":"Pund"},"convert_from_standard":[{"operator":"mul","value":"0.45359237"}],"symbol":"lb"}]'),
    ('Pressure', '{"ca_ES":"Pressió","da_DK":"Tryk","de_DE":"Druck","en_GB":"Pressure","en_NZ":"Pressure","en_US":"Pressure","es_ES":"Presión","fi_FI":"Paine","fr_FR":"Pression","it_IT":"Pressione","ja_JP":"圧力","pt_BR":"Pressão","ru_RU":"Давление","sv_SE":"Tryck"}', 'BAR', '[{"code":"CENTIBAR","labels":{"fr_FR":"Centibar","en_US":"Centibar"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"cbar"},{"code":"DECIBAR","labels":{"fr_FR":"Décibar","en_US":"Decibar"},"convert_from_standard":[{"operator":"mul","value":"0.1"}],"symbol":"dbar"},{"code":"KILOBAR","labels":{"fr_FR":"Kilobar","en_US":"Kilobar"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kbar"},{"code":"MEGABAR","labels":{"fr_FR":"Mégabar","en_US":"Megabar"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"Mbar"},{"code":"KILOPASCAL","labels":{"fr_FR":"Kilopascal","en_US":"Kilopascal"},"convert_from_standard":[{"operator":"mul","value":"0.01"}],"symbol":"kPa"},{"code":"MEGAPASCAL","labels":{"fr_FR":"Mégapascal","en_US":"Megapascal"},"convert_from_standard":[{"operator":"mul","value":"10"}],"symbol":"MPa"},{"code":"GIGAPASCAL","labels":{"fr_FR":"Gigapascal","en_US":"Gigapascal"},"convert_from_standard":[{"operator":"mul","value":"10000"}],"symbol":"GPa"},{"code":"BAR","labels":{"ca_ES":"Bar","da_DK":"Bar","de_DE":"Bar","en_GB":"Bar","en_NZ":"Bar","en_US":"Bar","es_ES":"Bar","fi_FI":"Baari","fr_FR":"Bar","it_IT":"Bar","ja_JP":"バー","pt_BR":"Bar","ru_RU":"Бар","sv_SE":"Bar"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"Bar"},{"code":"PASCAL","labels":{"ca_ES":"Pascal","da_DK":"Pascal","de_DE":"Pascal","en_GB":"Pascal","en_NZ":"Pascal","en_US":"Pascal","es_ES":"Pascal","fi_FI":"Pascal","fr_FR":"Pascal","it_IT":"Pascal","ja_JP":"パスカル","pt_BR":"Pascal","ru_RU":"Паскаль","sv_SE":"Pascal"},"convert_from_standard":[{"operator":"mul","value":"0.00001"}],"symbol":"Pa"},{"code":"HECTOPASCAL","labels":{"ca_ES":"hectopascal","da_DK":"HECTO Pascal","de_DE":"Hekto Pascal","en_GB":"hectopascal","en_NZ":"hecto Pascal","en_US":"hecto Pascal","es_ES":"hecto Pascal","fi_FI":"Hehtopascal","fr_FR":"HectoPascal","it_IT":"Hectopascal","ja_JP":"ヘクト パスカル","pt_BR":"hecto Pascal","ru_RU":"Гектопаскаль","sv_SE":"hektopascal"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"hPa"},{"code":"MILLIBAR","labels":{"ca_ES":"mil·libar","da_DK":"Milli Bar","de_DE":"Milli-Bar","en_GB":"millibar","en_NZ":"milli Bar","en_US":"milli Bar","es_ES":"milibar","fi_FI":"Millibaari","fr_FR":"Millibar","it_IT":"Millibar","ja_JP":"ミリ バー","pt_BR":"mili Bar","ru_RU":"Миллибар","sv_SE":"millibar"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mBar"},{"code":"ATM","labels":{"ca_ES":"Atmosfera","da_DK":"Atmosfære","de_DE":"Atmosphäre","en_GB":"Atmosphere","en_NZ":"Atmosphere","en_US":"Atmosphere","es_ES":"Atmósfera","fi_FI":"Ilmakehä","fr_FR":"Atmosphère","it_IT":"Atmosfera","ja_JP":"環境","pt_BR":"Atmosfera","ru_RU":"Атмосфера","sv_SE":"Atmosfär"},"convert_from_standard":[{"operator":"mul","value":"1.01325"}],"symbol":"atm"},{"code":"PSI","labels":{"ca_ES":"Lliura per polzada quadrada","da_DK":"Pund pr. kvadrattomme","de_DE":"Pfund pro Quadratzoll","en_GB":"Pound per square inch","en_NZ":"Pound per square inch","en_US":"Pound per square inch","es_ES":"Libra por pulgada cuadrada","fi_FI":"Paunaa per neliötuuma","fr_FR":"Livre par pouce carré","it_IT":"Libbra per pollice quadrato","ja_JP":"1 平方インチあたりのポンド","pt_BR":"Libra por polegada quadrada","ru_RU":"Фунт на квадратный дюйм","sv_SE":"Pund per kvadrattum"},"convert_from_standard":[{"operator":"mul","value":"0.0689476"}],"symbol":"PSI"},{"code":"TORR","labels":{"ca_ES":"Torr","da_DK":"Torr","de_DE":"Torr","en_GB":"Torr","en_NZ":"Torr","en_US":"Torr","es_ES":"Torr","fi_FI":"Torri","fr_FR":"Torr","it_IT":"Torr","ja_JP":"トル","pt_BR":"Torr","ru_RU":"Торр","sv_SE":"Torr"},"convert_from_standard":[{"operator":"mul","value":"0.00133322"}],"symbol":"Torr"},{"code":"MMHG","labels":{"ca_ES":"mil·límetre de Mercuri","da_DK":"millimeter kviksølv","de_DE":"Millimeter Quecksilbersäule","en_GB":"millimetre of mercury","en_NZ":"millimetre of Mercury","en_US":"millimeter of Mercury","es_ES":"milímetro de mercurio","fi_FI":"elohopeamillimetri","fr_FR":"Millimètre de mercure","it_IT":"Millimetro di mercurio","ja_JP":"水銀のミリメートル","pt_BR":"milímetro de mercúrio","ru_RU":"Миллиметр ртутного столба","sv_SE":"millimeter kvicksilver"},"convert_from_standard":[{"operator":"mul","value":"0.00133322"}],"symbol":"mmHg"}]'),
    ('Energy', '{"ca_ES":"Energia","da_DK":"Energi","de_DE":"Energie","en_GB":"Energy","en_NZ":"Energy","en_US":"Energy","es_ES":"Energía","fi_FI":"Energia","fr_FR":"Energie","it_IT":"Energia","ja_JP":"エネルギー","pt_BR":"Energia","ru_RU":"Энергия","sv_SE":"Energi"}', 'JOULE', '[{"code":"JOULE","labels":{"de_DE":"Joule","en_US":"joule","fr_FR":"joule"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"J"},{"code":"CALORIE","labels":{"de_DE":"Kalorien","en_US":"calorie","fr_FR":"calorie"},"convert_from_standard":[{"operator":"mul","value":"4.184"}],"symbol":"cal"},{"code":"KILOCALORIE","labels":{"ca_ES":"quilocalories","da_DK":"kilokalorie","de_DE":"Kilokalorien","en_GB":"kilocalorie","en_NZ":"kilocalorie","en_US":"kilocalorie","es_ES":"kilo caloría","fi_FI":"Kilokalori","fr_FR":"kilocalorie","it_IT":"kilocalorie","ja_JP":"キロカロリー","pt_BR":"quilocaloria","ru_RU":"килокалория","sv_SE":"kilokalori"},"convert_from_standard":[{"operator":"mul","value":"4184"}],"symbol":"kcal"},{"code":"KILOJOULE","labels":{"ca_ES":"kilojoule","da_DK":"kJ","de_DE":"Kilojoule","en_GB":"kilojoule","en_NZ":"kilojoule","en_US":"kilojoule","es_ES":"kilojulio","fi_FI":"kilojoule","fr_FR":"kilojoule","it_IT":"kilojoule","ja_JP":"キロジュール","pt_BR":"kilojoule","ru_RU":"килоджоуль","sv_SE":"kilojoule"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kJ"}]'),
    ('CaseBox', '{"ca_ES":"Embalatge","da_DK":"Emballage","de_DE":"Verpackung","en_GB":"Packaging","en_NZ":"Packaging","en_US":"Packaging","es_ES":"Embalaje","fi_FI":"Pakkaus","fr_FR":"Emballage","it_IT":"Imballo","ja_JP":"包装","pt_BR":"Embalagens","ru_RU":"Упаковка","sv_SE":"Förpackning"}', 'PIECE', '[{"code":"PIECE","labels":{"ca_ES":"Peça","da_DK":"Stykke","de_DE":"Stück","en_GB":"Piece","en_NZ":"Piece","en_US":"Piece","es_ES":"Pieza","fi_FI":"Kappale","fr_FR":"Unité","it_IT":"Pezzo","ja_JP":"作品","pt_BR":"Peça","ru_RU":"шт.","sv_SE":"Stycke"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"Pc"},{"code":"DOZEN","labels":{"ca_ES":"Dotzena","da_DK":"Dusin","de_DE":"Dutzend","en_GB":"Dozen","en_NZ":"Dozen","en_US":"Dozen","es_ES":"Docena","fi_FI":"Tusina","fr_FR":"Douzaine","it_IT":"Dozzina","ja_JP":"ダース","pt_BR":"Dúzia","ru_RU":"Дюжина","sv_SE":"Dussin"},"convert_from_standard":[{"operator":"mul","value":"12"}],"symbol":"Dz"}]'),
    ('Brightness', '{"de_DE":"Helligkeit","en_US":"Brightness","fr_FR":"Luminosité"}', 'LUMEN', '[{"code":"LUMEN","labels":{"de_DE":"Lumen","en_US":"Lumen","fr_FR":"Lumen"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"lm"},{"code":"NIT","labels":{"de_DE":"Nit","en_US":"Nit","fr_FR":"Nit"},"convert_from_standard":[{"operator":"mul","value":"0.2918855809"}],"symbol":"nits"}]'),
    ('Force', '{"de_DE":"Kraft","fr_FR":"Force","en_US":"Force"}', 'NEWTON', '[{"code":"MILLINEWTON","labels":{"fr_FR":"Millinewton","en_US":"Millinewton"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mN"},{"code":"NEWTON","labels":{"fr_FR":"Newton","en_US":"Newton"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"N"},{"code":"KILONEWTON","labels":{"fr_FR":"Kilonewton","en_US":"Kilonewton"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kN"},{"code":"MEGANEWTON","labels":{"fr_FR":"Méganewton","en_US":"Meganewton"},"convert_from_standard":[{"operator":"mul","value":"1000000"}],"symbol":"MN"}]'),
    ('Angle', '{"de_DE":"Winkel","fr_FR":"Angle","en_US":"Angle"}', 'RADIAN', '[{"code":"RADIAN","labels":{"fr_FR":"Radian","en_US":"Radian"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"rad"},{"code":"MILLIRADIAN","labels":{"fr_FR":"Milliradian","en_US":"Milliradian"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"mrad"},{"code":"MICRORADIAN","labels":{"fr_FR":"Microradian","en_US":"Microradian"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"µrad"},{"code":"DEGREE","labels":{"fr_FR":"Degré","en_US":"Degree"},"convert_from_standard":[{"operator":"mul","value":"0.01745329"}],"symbol":"°"},{"code":"MINUTE","labels":{"fr_FR":"Minute","en_US":"Minute"},"convert_from_standard":[{"operator":"mul","value":"0.0002908882"}],"symbol":"\'"},{"code":"SECOND","labels":{"fr_FR":"Seconde","en_US":"Second"},"convert_from_standard":[{"operator":"mul","value":"0.000004848137"}],"symbol":"\\\""},{"code":"GON","labels":{"fr_FR":"Gon","en_US":"Gon"},"convert_from_standard":[{"operator":"mul","value":"0.01570796"}],"symbol":"gon"},{"code":"MIL","labels":{"fr_FR":"Mil","en_US":"Mil"},"convert_from_standard":[{"operator":"mul","value":"0.0009817477"}],"symbol":"mil"},{"code":"REVOLUTION","labels":{"fr_FR":"Révolution","en_US":"Revolution"},"convert_from_standard":[{"operator":"mul","value":"6.283185"}],"symbol":"rev"}]'),
    ('Capacitance', '{"de_DE":"Elektrische Kapazität","fr_FR":"Capacité électrique","en_US":"Capacitance"}', 'FARAD', '[{"code":"ATTOFARAD","labels":{"fr_FR":"Attofarad","en_US":"Attofarad"},"convert_from_standard":[{"operator":"div","value":"1000000000000000000"}],"symbol":"aF"},{"code":"PICOFARAD","labels":{"fr_FR":"Picofarad","en_US":"Picofarad"},"convert_from_standard":[{"operator":"div","value":"1000000000000"}],"symbol":"pF"},{"code":"NANOFARAD","labels":{"fr_FR":"Nanofarad","en_US":"Nanofarad"},"convert_from_standard":[{"operator":"div","value":"1000000000"}],"symbol":"nF"},{"code":"MICROFARAD","labels":{"fr_FR":"Microfarad","en_US":"Microfarad"},"convert_from_standard":[{"operator":"div","value":"1000000"}],"symbol":"µF"},{"code":"MILLIFARAD","labels":{"fr_FR":"Millifarad","en_US":"Millifarad"},"convert_from_standard":[{"operator":"div","value":"1000"}],"symbol":"mF"},{"code":"FARAD","labels":{"fr_FR":"Farad","en_US":"Farad"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"F"},{"code":"KILOFARAD","labels":{"fr_FR":"Kilofarad","en_US":"Kilofarad"},"convert_from_standard":[{"operator":"mul","value":"1000"}],"symbol":"kF"}]'),
    ('VolumeFlow', '{"de_DE":"Volumenstrom","fr_FR":"Débit volumétrique","en_US":"Volume flow"}', 'CUBIC_METER_PER_SECOND', '[{"code":"CUBIC_METER_PER_SECOND","labels":{"fr_FR":"Mètre cube par seconde","en_US":"Cubic meter per second"},"convert_from_standard":[{"operator":"mul","value":"1"}],"symbol":"m³/s"},{"code":"CUBIC_METER_PER_MINUTE","labels":{"fr_FR":"Mètre cube par minute","en_US":"Cubic meter per minute"},"convert_from_standard":[{"operator":"mul","value":"60"}],"symbol":"m³/min"},{"code":"CUBIC_METER_PER_HOUR","labels":{"fr_FR":"Mètre cube par heure","en_US":"Cubic meter per hour"},"convert_from_standard":[{"operator":"mul","value":"3600"}],"symbol":"m³/h"},{"code":"CUBIC_METER_PER_DAY","labels":{"fr_FR":"Mètre cube par jour","en_US":"Cubic meter per day"},"convert_from_standard":[{"operator":"mul","value":"86400"}],"symbol":"m³/d"},{"code":"MILLILITER_PER_SECOND","labels":{"fr_FR":"Millilitre par seconde","en_US":"Milliliter per second"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"ml/s"},{"code":"MILLILITER_PER_MINUTE","labels":{"fr_FR":"Millilitre par minute","en_US":"Milliliter per minute"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"div","value":"60"}],"symbol":"ml/min"},{"code":"MILLILITER_PER_HOUR","labels":{"fr_FR":"Millilitre par heure","en_US":"Milliliter per hour"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"div","value":"3600"}],"symbol":"ml/h"},{"code":"MILLILITER_PER_DAY","labels":{"fr_FR":"Millilitre par jour","en_US":"Milliliter per day"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"div","value":"86400"}],"symbol":"ml/d"},{"code":"CUBIC_CENTIMETER_PER_SECOND","labels":{"fr_FR":"Centimètre cube par seconde","en_US":"Cubic centimeter per second"},"convert_from_standard":[{"operator":"mul","value":"0.000001"}],"symbol":"cm³/s"},{"code":"CUBIC_CENTIMETER_PER_MINUTE","labels":{"fr_FR":"Centimètre cube par minute","en_US":"Cubic centimeter per minute"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"mul","value":"60"}],"symbol":"cm³/min"},{"code":"CUBIC_CENTIMETER_PER_HOUR","labels":{"fr_FR":"Centimètre cube par heure","en_US":"Cubic centimeter per hour"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"mul","value":"3600"}],"symbol":"cm³/h"},{"code":"CUBIC_CENTIMETER_PER_DAY","labels":{"fr_FR":"Centimètre cube par jour","en_US":"Cubic centimeter per day"},"convert_from_standard":[{"operator":"mul","value":"0.000001"},{"operator":"mul","value":"86400"}],"symbol":"cm³/d"},{"code":"CUBIC_DECIMETER_PER_MINUTE","labels":{"fr_FR":"Décimètre cube par minute","en_US":"Cubic decimeter per minute"},"convert_from_standard":[{"operator":"mul","value":"0.001"},{"operator":"mul","value":"60"}],"symbol":"dm³/min"},{"code":"CUBIC_DECIMETER_PER_HOUR","labels":{"fr_FR":"Décimètre cube par heure","en_US":"Cubic decimeter per hour"},"convert_from_standard":[{"operator":"mul","value":"0.001"},{"operator":"mul","value":"3600"}],"symbol":"dm³/h"},{"code":"LITER_PER_SECOND","labels":{"fr_FR":"Litre par seconde","en_US":"Liter per second"},"convert_from_standard":[{"operator":"mul","value":"0.001"}],"symbol":"l/s"},{"code":"LITER_PER_MINUTE","labels":{"fr_FR":"Litre par minute","en_US":"Liter per minute"},"convert_from_standard":[{"operator":"mul","value":"0.001"},{"operator":"div","value":"60"}],"symbol":"l/min"},{"code":"LITER_PER_HOUR","labels":{"fr_FR":"Litre par heure","en_US":"Liter per hour"},"convert_from_standard":[{"operator":"mul","value":"0.001"},{"operator":"div","value":"3600"}],"symbol":"l/h"},{"code":"LITER_PER_DAY","labels":{"fr_FR":"Litre par jour","en_US":"Liter per day"},"convert_from_standard":[{"operator":"mul","value":"0.001"},{"operator":"div","value":"86400"}],"symbol":"l/d"},{"code":"KILOLITER_PER_HOUR","labels":{"fr_FR":"Kilolitre par heure","en_US":"Kiloliter per hour"},"convert_from_standard":[{"operator":"mul","value":"3600"}],"symbol":"kl/h"}]');
SQL;

        $this->connection->exec($sql);
    }

    private function createMeasurementTable(): void
    {
        $sql = <<<SQL
DROP TABLE IF EXISTS `akeneo_measurement`;
CREATE TABLE `akeneo_measurement` (
  `code` varchar(100) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL UNIQUE,
  `labels` JSON NOT NULL,
  `standard_unit` varchar(100) NOT NULL,
  `units` JSON NOT NULL,
  PRIMARY KEY (`code`)
) ENGINE=InnoDB AUTO_INCREMENT=169 DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci ROW_FORMAT=DYNAMIC;
SQL;

        $this->connection->exec($sql);
    }
}
